"use strict";

var toolbarData = {
  items: [
      {
          type: "button",
          file: "icons/center.svg",
          tooltip: "Center",
          name: "center"
      },
      {
          type: "separator"
      },
      {
          type: "button",
          file: "icons/out.svg",
          tooltip: "Zoom Out",
          name: "zoom-out"
      },
      {
          type: "button",
          file: "icons/reset.svg",
          tooltip: "Reset Zoom",
          name: "zoom-reset"
      },
      {
          type: "button",
          file: "icons/in.svg",
          tooltip: "Zoom In",
          name: "zoom-in"
      },
      {
          type: "button",
          file: "icons/fit.svg",
          tooltip: "Zoom to Fit",
          name: "zoom-fit"
      },
      {
          type: "icon",
          file: "icons/zoom.svg"
      },
      {
          type: "label",
          value: "100%"
      },
  ]
};

var toolbarItems = {
    center:    toolbarData.items[0],
    zoomIn:    toolbarData.items[2],
    zoomReset: toolbarData.items[3],
    zoomOut:   toolbarData.items[4],
    zoomFit:   toolbarData.items[5],
    zoomLabel: toolbarData.items[7],
};

var surface = window.document.querySelector(".img-container");
var image   = window.document.querySelector(".img-container > img");
var canvas  = window.document.querySelector(".img-container > canvas");

var zoom = 1.0;

var originalWidth  = 0;
var originalHeight = 0;
originalWidth  = image.clientWidth;
originalHeight = image.clientHeight;

// Center image
image.style.left = "0px";
image.style.top  = "0px";
canvas.style.left = "0px";
canvas.style.top = "0px";

// Capture the image dimensions if it loads after the page
image.onload = function() {
    originalWidth  = image.clientWidth;
    originalHeight = image.clientHeight;

    // Say we are "loaded"
    window.parent.postMessage({
        'name': 'updateStatus',
        'data': "loaded"
    }, '*');
};

const renderPCX = function(buffer) {
    // Decode the pcx data
    const context = canvas.getContext('2d');

    const pcx = new PCX(buffer);
    const pcxData = pcx.decode(context);

    const width  = pcxData.width,
          height = pcxData.height;

    // resize canvas
    context.canvas.width  = width;
    context.canvas.height = height;

    originalWidth  = width;
    originalHeight = height;

    var imageData = new ImageData(width, height);
    imageData.data.set(pcxData.pixelArray);

    context.putImageData(imageData, 0, 0);

    // Say we are "loaded"
    window.parent.postMessage({
        'name': 'updateStatus',
        'data': "loaded"
    }, '*');
}

// When the page has JavaScript, it will pan using the mouse/touch
// and not need the scrollbars.
surface.style.overflow = "hidden";

surface.addEventListener('mousedown', function(event) {
    event.preventDefault();
    event.stopPropagation();

    var startX = event.pageX;
    var startY = event.pageY;

    var originX = parseInt(image.style.left) || 0;
    var originY = parseInt(image.style.top)  || 0;

    if (image.hasAttribute('hidden')) {
        originX = parseInt(canvas.style.left) || 0;
        originY = parseInt(canvas.style.top) || 0;
    }

    var dragEvent = function(event) {
        var deltaX = event.pageX - startX;
        var deltaY = event.pageY - startY;

        image.style.left = (originX + deltaX) + "px";
        image.style.top  = (originY + deltaY) + "px";

        canvas.style.left = (originX + deltaX) + "px";
        canvas.style.top  = (originY + deltaY) + "px";
    };

    let view = ((image || window).ownerDocument || window).defaultView || window;
    view.addEventListener('mouseup', function(event) {
        view.removeEventListener('mousemove', dragEvent);
    });
    view.addEventListener('mousemove', dragEvent);
});

function wheelEvent(event) {
    var delta = event.wheelDeltaY || (event.deltaY * -15);

    if (delta) {
        zoom += (delta / 120) * 0.25;
        doZoom();
    }
};

surface.addEventListener('mousewheel', wheelEvent);
surface.addEventListener('wheel', wheelEvent);

var doZoom = function() {
    if (zoom < 0.25) {
        zoom = 0.25;
    }

    if (!isFinite(zoom)) {
        zoom = 1.0;
    }

    canvas.style.transform = "scale(" + zoom + ")";
    canvas.style.transformOrigin = "middle middle";

    image.style.transform = "scale(" + zoom + ")";
    image.style.transformOrigin = "middle middle";
    
    toolbarItems.zoomLabel.value = Math.round(zoom * 100) + "%";
    window.parent.postMessage({
        'name': 'updateToolbar',
        'data': toolbarData
    }, '*');
}

/* Copies a touch event */
function copyTouch(touch) {
    return { identifier: touch.identifier, pageX: touch.pageX, pageY: touch.pageY };
}

function ongoingTouchIndexById(ongoingTouches, idToFind) {
    for (var i = 0; i < ongoingTouches.length; i++) {
        var id = ongoingTouches[i].identifier;

        if (id == idToFind) {
            return i;
        }
    }
    return -1;    // not found
}

/* Handle the 'touchstart' event for touch screen devices.  */
var touchStartEvent = function(event) {
    var element = this;

    // Suppress the mouse click
    event.preventDefault();

    element.originX = parseInt(image.style.left) || 0;
    element.originY = parseInt(image.style.top)  || 0;

    if (image.hasAttribute('hidden')) {
        element.originX = parseInt(canvas.style.left) || 0;
        element.originY = parseInt(canvas.style.top) || 0;
    }

    var touches = event.changedTouches;
    for (var i = 0; i < touches.length; i++) {
        var touch = touches[i];

        element.ongoingTouches.push(copyTouch(touch));
    };

    if (touches.length == 1) {
        var eventPoint = [touches[0].pageX, touches[0].pageY];
        var eventX = eventPoint[0];
        var eventY = eventPoint[1];

        element.startX = eventX;
        element.startY = eventY;
    }
    if (touches.length == 2) {
        var eventPoint = [touches[0].pageX, touches[0].pageY];
        var eventX = eventPoint[0];
        var eventY = eventPoint[1];
    }
};

/* Handle the 'touchmove' event for touch screen devices.  */
var touchMoveEvent = function(event) {
    var element = this;

    // Suppress the mouse move
    event.preventDefault();

    var touches = event.changedTouches;
    if (touches.length == 1) {
        // One finger (Move workflow)
        var eventPoint = [touches[0].pageX, touches[0].pageY];
        var eventX = eventPoint[0];
        var eventY = eventPoint[1];

        var deltaX = eventX - element.startX;
        var deltaY = eventY - element.startY;

        image.style.left = (element.originX + deltaX) + "px";
        image.style.top  = (element.originY + deltaY) + "px";
    }
    else if (touches.length == 1) {
    }
};

/* Handle the 'touchend' event for touch screen devices.
*/
var touchEndEvent = function(event) {
    var element = this;

    // Suppress the mouse click
    event.preventDefault();

    var touches = event.changedTouches;
    for (var i = 0; i < touches.length; i++) {
        var touch = touches[i];

        var idx = ongoingTouchIndexById(element.ongoingTouches, touch.identifier);
        if (idx >= 0) {
            element.ongoingTouches.splice(idx, 1);
        }
    }
};

surface.ongoingTouches = [];
surface.addEventListener('touchstart',  touchStartEvent);
surface.addEventListener('touchmove',   touchMoveEvent);
surface.addEventListener('touchend',    touchEndEvent);
surface.addEventListener('touchcancel', touchEndEvent);

// Handle OCCAM events
window.addEventListener('message', function(event) {
    if (event.data.name === 'updateConfiguration') {
    }
    else if (event.data.name === 'updateToolbar') {
        if (event.data.data.name === 'center') {
            image.style.left = "0px";
            image.style.top  = "0px";
            canvas.style.left = "0px";
            canvas.style.top  = "0px";
        }
        else if (event.data.data.name === 'zoom-reset') {
            zoom = 1.0;
            doZoom();
        }
        else if (event.data.data.name === 'zoom-out') {
            zoom -= 0.25;
            doZoom();
        }
        else if (event.data.data.name === 'zoom-in') {
            zoom += 0.25;
            doZoom();
        }
        else if (event.data.data.name === 'zoom-fit') {
            var zoomX = surface.clientWidth  / originalWidth;
            var zoomY = surface.clientHeight / originalHeight;

            image.style.left = "0px";
            image.style.top  = "0px";
            canvas.style.left = "0px";
            canvas.style.top  = "0px";

            zoom = Math.min(zoomX, zoomY);
            doZoom();
        }
    }
    else if (event.data.name === 'updateStatus') {
        if (event.data.data == "ready") {
            // Maximize the widget
            window.parent.postMessage({
                'name': 'updateSize',
                'data': {
                    'height': "100%"
                }
            }, '*');

            // Say we are "loading"
            window.parent.postMessage({
                'name': 'updateStatus',
                'data': "loading"
            }, '*');

            // Initialize toolbar
            window.parent.postMessage({
                'name': 'updateToolbar',
                'data': toolbarData
            }, '*');

            // Request input data
            window.parent.postMessage({
                'name': 'updateInput'
            }, '*');
        }
    }
    else if (event.data.name === 'updateInput') {
        var objectInfo = event.data.data;

        var inputInfo = objectInfo;

        // Say we are "loading"
        window.parent.postMessage({
            'name': 'updateStatus',
            'data': "loading"
        }, '*');

        // Get the filename to pull the data
        if (inputInfo) {
            var filename = objectInfo.file;
            var url = objectInfo.url;

            // Determine canvas or image
            var renderFunction = null;
            if (filename.substring(filename.length - 4) == ".pcx") {
                // PCX image
                image.setAttribute('hidden', '');
                canvas.removeAttribute('hidden');
                renderFunction = renderPCX;
            }
            else {
                image.removeAttribute('hidden');
                canvas.setAttribute('hidden', '');
                image.src = url;
            }

            // Pull down file and render
            if (renderFunction) {
                // Pull the data
                fetch(url, {
                    credentials: 'include'
                }).then(function(response) {
                    return response.arrayBuffer()
                }).then(function(data) {
                    renderFunction(data);
                });
            }
        }
        else {
            image.src = "logo_large.svg";
        }
    }
});

// We are ready.

// Say we are "loading"
window.parent.postMessage({
    'name': 'updateStatus',
    'data': "loading"
}, '*');

// Initialize toolbar
window.parent.postMessage({
    'name': 'updateToolbar',
    'data': toolbarData
}, '*');

// Request configuration data.
window.parent.postMessage({
    'name': 'updateConfiguration'
}, '*');

// Request input file (if any)
window.parent.postMessage({
    'name': 'updateInput'
}, '*');
