/************************************************************************
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 ************************************************************************/

#define	MAXTRACKS	10

// 	Registers for the Sound Blaster card - (add I/O base)
#define	sbReset			0x06	// W
#define	sbFMStatus		0x08	// R
#define	sbFMAddr		0x08	// W
#define	sbFMData		0x09	// W
#define	sbReadData		0x0a	// R
#define	sbWriteCmd		0x0c	// W
#define	sbWriteData		0x0c	// W
#define	sbWriteStat		0x0c	// R
#define	sbDataAvail		0x0e	// R

//	Registers for the Sound Blaster Pro card - (add I/O base)
#define	sbpLFMStatus	0x00	// R
#define	sbpLFMAddr		0x00	// W
#define	sbpLFMData		0x01	// W
#define	sbpRFMStatus	0x02	// R
#define	sbpRFMAddr		0x02	// W
#define	sbpRFMData		0x03	// W
#define	sbpMixerAddr	0x04	// W
#define	sbpMixerData	0x05	// RW
#define	sbpCDData		0x10	// R
#define	sbpCDCommand	0x10	// W
#define	sbpCDStatus		0x11	// R
#define	sbpCDReset		0x12	// W

// 	Registers for the AdLib card
#define	alFMStatus		0x388	// R
#define	alFMAddr		0x388	// W
#define	alFMData		0x389	// W

//	Register addresses
// Operator stuff
#define	alAVEKM			0x20
#define	alKOutLev		0x40
#define	alAttDec		0x60
#define	alSusRel		0x80
#define	alWave			0xe0
// Channel stuff
#define	alFreqL			0xa0
#define	alFreqH			0xb0
#define	alFeedCon		0xc0
// Global stuff
#define	alAMVDR			0xbd

typedef	struct {
			byte 	mAVEKM,cAVEKM,
					mKOutLev,cKOutLev,
					mAttDec,cAttDec,
					mSusRel,cSusRel,
					mWave,cWave,
					nFeedCon
					//,unused[5]	// 16 bytes bound
		} TInstrument;

typedef struct {
			TInstrument	*inst;
			char		percussive;
			word		far *seq;
			longword	nextevent;
		} TTrack;

byte mp_StatusReg();
/* Returns the FM status register:

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
   IRQ        FLAG                                               
           T1     T2                                            
 
*/

void mp_InitCard();
// This must be called to initialize the FM synthesizer chips.

void mp_SetTimer1(byte count);
/* Timer1 is an 8 bit presettable counter.  It has a resolution of 80usec.
  Count is the preset value to be loaded into the counter.  If an overflow
  occurs, the Timer1 flag in the status register is set, and the preset
  value is reloaded into the counter.  In addition to normal timer func-
  tions, Timer1 is also used for control of composite speech synthesis. */

void mp_SetTimer2(byte count);
// Timer2 is similar to Timer1 except that its resolution is 320usec.

void mp_TimerControl(byte control);
/* This procedure is used to start, stop and control flags of the timers.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
   IRQ         MASK                               START/STOP    
   RST     T1     T2                            T1      T2    
 

  D0 (Start/Stop Timer1)
	When this bit is 1, the preset value is loaded into Timer1 and
	counting starts.  When this bit is 0, Timer1 does not operate.
  D1 (Start/Stop Timer2)
	Performs the same function as D0 for Timer2.
  D5 (Mask Timer2)
	When this bit is 1, the flag for Timer2 in the status register is
	masked (always 0).
  D6 (Mask Timer1)
	Bit mask flag for Timer1.
  D7 (IRQ-RESET)
	When set to 1, the IRQ and Timer flags in the status register are
	reset to 0.  This bit then resets itself to 0. */

void mp_CSMMode_KbdSplit(byte mks);
/* This procedure causes the synthesizer to operate in music mode or speech
  synthesis mode and determines the keyboard split for keyboard scaling of
  rate.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
   CSM     SEL                                                   
 

  D6 (NOTE SELECT)
	This bit controls the split point of the keyboard.	When 0, the key-
	board split is the second bit from the most significant bit of the
	F-Number.  The MSB of the F-Number is used when 1.

	NOTE SELECT = 0
 Ŀ
    OCTAVE      0      1      2      3      4      5      6      7   
 Ĵ
  BLOCK DATA    0      1      2      3      4      5      6      7   
 Ĵ
  F-NUMBER 9    1      1      1      1      1      1      1      1   
 Ĵ
  F-NUMBER 8  0  1  0  1  0  1  0  1  0  1  0  1  0  1  0  1 
 Ĵ
  SPLIT NUM   0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 
 

	NOTE SELECT = 1
 Ŀ
    OCTAVE      0      1      2      3      4      5      6      7   
 Ĵ
  BLOCK DATA    0      1      2      3      4      5      6      7   
 Ĵ
  F-NUMBER 9  0  1  0  1  0  1  0  1  0  1  0  1  0  1  0  1 
 Ĵ
  SPLIT NUM   0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 
 

  D7 (CSM)
	The composite sine wave speech synthesis mode is selected when 1.  All
	channels must be off when this mode is selected. */

void mp_AM_VIB_EG_KSR_multiple(byte instnum, byte modulator, byte carrier);
/* This procedure controls several parameters affecting the timbre of the
  output signal.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
    AM     VIB    EG     KSR          MULTIPLE                 
 

  D0-D3 (MULTIPLE)
	The basic FM equation with the multiplication factor included is:

	  F(t) = A sin (Mc Wc t + I sin (Mm Wm t))

	where A is the output amplitude
	  I is the modulation index (amplitude of Modulator)
	  Wc is the carrier frequency
	  Wm is the modulator frequency
	  Mc is the multiplication factor for Carrier cell
	  Mm is the multiplication factor for Modulator cell

	The relationship between MULTIPLE and multiplication factor is:
	Ŀ  Ŀ
	 MULTIPLE  MULTIPLICATION    MULTIPLE  MULTIPLICATION 
	               FACTOR                      FACTOR     
	Ĵ  Ĵ
	     0           0.5             8            8       
	     1             1             9            9       
	     2             2             A           10       
	     3             3             B           10       
	     4             4             C           12       
	     5             5             D           12       
	     6             6             E           15       
	     7             7             F           15       
	  

  D4 (KEY SCALING RATE)
	When this bit is set to 1, the envelope length is gradually shortened
	as higher notes on the keyboard are played.
  D5 (ENVELOPE TYPE)
	A diminishing sound is selected when this bit is 0 and a continuous
	sound when it is 1.
  D6 (VIBRATO)
	Setting this bit to 1 cause a vibrato effect to be applied to the
	operator cell.
  D7 (AMPLITUDE MODULATION)
	Setting this bit to 1 causes tremolo to be applied to operator cell. */

void mp_KSL_TotalLevel(byte instnum, byte modulator, byte carrier);
/* This procedure controls the attenuation of the operator cells' output
  signal.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
       KSL                          TOTAL LEVEL                   
 

  D0-D5 (TOTAL LEVEL)
	The TOTAL LEVEL attenuates the operator's output.  Varying the output
	level of an operator functioning as a carrier results in a change in
	the overall level of the sound contributed to the voice by that operator.
	This is how the volume changes.  Attenuating the output from a modulator
	will change the frequency spectrum produces by the carrier.  TOTAL LEVEL
	has a range of 0 to 63.  Maximum attenuation is 47.25 dB.  Degree of
	attenuation is given be the following formula.

	  Attenuation = TOTAL LEVEL * 0.75 dB

  D6-D7 (KEYBOARD SCALING LEVEL)
	The KSL produces a gradual decrease in note output level towards higher
	pitch notes.  The degree of attenuation is controlled as follows:
	Ŀ
	 D7  D6  Degree of Attenuation 
	Ĵ
	  0   0            0           
	  1   0        1.5dB/OCT       
	  0   1         3dB/OCT        
	  1   1         6dB/OCT        
	
*/

void mp_Attack_Decay(byte instnum, byte modulator, byte carrier);
/* The ATTACK RATE is the rising time for the sound.  The DECAY RATE is the
  diminishing time after the attack.  These rates range from 0 to 15.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
           ATTACK RATE                      DECAY RATE            
    2^3     2^2     2^1     2^0     2^3     2^2     2^1     2^0   
 
*/

void mp_Sustain_Release(byte instnum, byte modulator, byte carrier);
/* For continuing sounds, the SUSTAIN LEVEL gives the point of change where
  the attenuated sound in the decay mode changes to a sound having a con-
  stant level.	For diminishing sounds, the sustain level gives the point
  where the decay mode changes to the release mode.

  For continuing sounds, the RELEASE RATE defines the rate at which the
  sound disappears after the notes is turned off.  For diminishing sounds,
  release rate indicates the attenuation after the sustain level is reached.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
          SUSTAIN LEVEL                    RELEASE RATE           
    24dB    12dB    6dB     3dB     2^3     2^2     2^1     2^0   
 
*/

void mp_Block_FNum(byte instnum, byte modulator, byte carrier);
/* This procedure determines the octave and frequency of the output signal.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
                           F-NUMBER (L)                            
 Ĵ
                  KEYON         BLOCK            F-NUMBER (H)   
 

  D0-D7 (F-NUMBER L) & D0-D1 (F-NUMBER H)
	The F-Number is a 10 bit number that determines the frequency of the
	output signal.
  D2-D4 (BLOCK)
	These 3 buts determine the octave of the output signal.
  D5 (KEYON)
	When this be is 1, the channels is ON and voiced.  When 0, the channel
	is OFF.

  The relationship among F-NUMBER, BLOCK and output frequency is as follows:

	FREQUENCY = 50000 * F-NUMBER * 2^BLOCK-20

  FREQUENCY FOR BLOCK 4
  Ŀ
                              FNUM H           F-NUMBER L            
   NOTE  FREQUENCY  F-NUMBER Ĵ
                              D1  D0  D7  D6  D5  D4  D3  D2  D1  D0 
  Ĵ
    C#     277.2   	  363	  0   1   0	1   1	0   1	0   1	1 
    D	    293.7   	  385	  0   1   1	0   0	0   0	0   0	1 
    D#     311.1   	  408	  0   1   1	0   0	1   1	0   0	0 
    E	    329.6   	  432	  0   1   1	0   1	1   0	0   0	0 
    F	    349.2   	  458	  0   1   1	1   0	0   1	0   1	0 
    F#     370.0   	  485	  0   1   1	1   1	0   0	1   0	1 
    G	    392.0   	  514	  1   0   0	0   0	0   0	0   1	0 
    G#     415.3   	  544	  1   0   0	0   1	0   0	0   0	0 
    A	    440.0   	  577	  1   0   0	1   0	0   0	0   0	1 
    A#     466.2   	  611	  1   0   0	1   1	0   0	0   1	1 
    B	    493.9   	  647	  1   0   1	0   0	0   0	1   1	1 
    C	    523.3   	  686	  1   0   1	0   1	0   1	1   1	0 
  
*/

void mp_AMVIBDepth_Rhythm(byte data);
/* This allows for control of amplitude modulation and vibrato depth, selec-
  tion of rhythm mode, and ON/OFF control of various rhythm instruments.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
       DEPTH       RHY-  BASS   SNARE   TOM    TOP     HI    
    AM     VIB    THM   DRUM   DRUM    TOM    CYM     HAT   
 

  D0-D5 (RHYTHM)
	The synthesizer is set to rhythm sound mode when D5 is 1.  D0-D4 allow
	for ON/OFF control of the various rhythm instruments.  The last 3 FM
	channels must be off.  Slots 13-18 correspond to the rhythm sound shown
	below.  Data such as rate must be input as a value appropriate to each
	rhythm sound.
	Ŀ
	 PERCUSSION INSTRUMENT  OPERATOR CELL 
	Ĵ
	       Bass Drum             13,16    
	       Hi Hat                14       
	       Tom Tom               15       
	       Snare Drum            17       
	       Top Cymbal            18       
	
  D6 (VIB DEPTH)
	The vibrato depth is 14 cent when 1, 7 cent when 0.
  D7 (AM DEPTH)
	The tremolo depth is 4.8dB when 1, 1dB when 0.
*/

void mp_WaveSelect(byte instnum, byte modulator, byte carrier);
/* When Wave Select is on, the output sine wave is distorted as shown below.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
                                                     WAVE SEL     
 

	Ŀ
	 D1  D0    Waveform   
	Ĵ
	          Ŀ        
	                  
	  0   0  Ŀ  
	                  
	                  
              Ŀ        
      0   1           
	           
	          Ŀ  Ŀ  
	  1   0       
	          
	          Ŀ   Ŀ   
	  1   1         
	           
	
*/

void mp_Feedback_Connect(byte instnum, byte data);
/* This procedure determines the modulation factor for self-feedback and
  the type of FM modulation.

 Ŀ
     D7      D6      D5      D4      D3      D2      D1      D0    
 Ĵ
                                         FEEDBACK          CON   
 

  D0 (CONNECTION)
    Connection controls the manner in which the two operator cells are
	connected.  The FM modulation mode is selected when this bit is 0.
    The two operator cells are connected in parallel for sine wave output
    in the composite mode when this bit is 1.
  D1-D3 (FEEDBACK)
	Gives the modulation factor for feedback FM modulation of the first
	operator cell.

Ŀ
  FEEDBACK     0      1      2      3      4      5      6      7   
Ĵ
 MODULATION                                                         
   FACTOR      0    II/16  II/8   II/4   II/2    II     2II    4II  

*/

void mp_LoadInst(byte instnum, TInstrument &instdata);
/* This procedure loads an instrument of InstType into the FM synthesizer.
  The instrument data starts at offset $24 of an SBI file.
*/

void mp_NoteOn(byte instnum, byte octave, byte note);
/* The procedure plays a note (1-12) in an octave (0-7) on channel (1-9).
  Note 1 is C#.
*/

void mp_NoteOff(byte instnum, byte octave, byte note);
/* The procedure stops a note (1-12) in an octave (0-7) on channel (1-9).
  (In this version, octave and note are ignored.)
*/

// *******************************************************

#include "std.hpp"

char mp_mactive = 0,
	mp_mplaying = 0;
int mp_mmemidx = -1;
word mp_mcurpos = 0;

// This table maps channel numbers to carrier and modulator op cells
static	byte			carriers[9]  = { 3, 4, 5,11,12,13,19,20,21},
						modifiers[9] = { 0, 1, 2, 8, 9,10,16,17,18},
// This table maps percussive voice numbers to op cells
						pcarriers[5]  = {19,0xff,0xff,0xff,0xff},
						pmodifiers[5] = {16,  17,  18,  20,  21};
byte choffs[9] = {0x01,0x02,0x03,0x09,0x0a,0x0b,0x11,0x12,0x13};
word fnums[12] = {363,385,408,432,458,485,514,544,577,611,647,686};

const TInstrument Inst[10] = {
{0x21,0x11,0x4C,0x00,0xF1,0xF2,0x63,0x72,         //  Instrument 0
 0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00},
{0xA5,0xB1,0xD2,0x80,0x81,0xF1,0x03,0x05,         //  Instrument 1
 0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00},
{0x72,0x62,0x1C,0x05,0x51,0x52,0x03,0x13,         //  Instrument 2
 0x00,0x00,0x0E,0x00,0x00,0x00,0x00,0x00},
{0x11,0x01,0x8A,0x40,0xF1,0xF1,0x11,0xB3,         //  Instrument 3
 0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00},
{0x21,0x11,0x11,0x00,0xA3,0xC4,0x43,0x22,         //  Instrument 4
 0x02,0x00,0x0D,0x00,0x00,0x00,0x00,0x00},
{0x31,0xA1,0x1C,0x80,0x41,0x92,0x0B,0x3B,         //  Instrument 5
 0x00,0x00,0x0E,0x00,0x00,0x00,0x00,0x00},
{0x71,0x62,0xC5,0x05,0x6E,0x8B,0x17,0x0E,         //  Instrument 6
 0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00},
{0x41,0x91,0x83,0x00,0x65,0x32,0x05,0x74,         //  Instrument 7
 0x00,0x00,0x0A,0x00,0x00,0x00,0x00,0x00},
{0x32,0x16,0x87,0x80,0xA1,0x7D,0x10,0x33,         //  Instrument 8
 0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00},
{0x01,0x13,0x8D,0x00,0x51,0x52,0x53,0x7C,         //  Instrument 9
 0x01,0x00,0x0C,0x00,0x00,0x00,0x00,0x00}};

void al_out(byte port, byte data) {
asm	{
	//pushf
	//cli
	MOV	DX,	0x388
	MOV	AL, port
	OUT	DX, AL
	IN	AL, DX
	IN	AL, DX
	IN	AL, DX
	IN	AL, DX
	IN	AL, DX
	IN	AL, DX
	INC	DX
	MOV	AL, data
	OUT	DX, AL
	//popf
	DEC	DX

	MOV CX, 35	// 35
} c1: asm {
	IN AL, DX
	LOOP c1
}

byte mp_StatusReg() {
	return inportb(0x388);
}

void mp_InitCard() {
  //outportb(0x388,1);
	al_out(1,0);
  //delay(1);
  //outportb(0x389,0);
  //delay(1);
}

void mp_SetTimer1(byte count) {
	al_out(2,count);
}

void mp_SetTimer2(byte count) {
	al_out(3,count);
}

void mp_TimerControl(byte control) {
	al_out(4,control);
}

void mp_CSMMode_KbdSplit(byte mks) {
	al_out(8,mks);
}

void mp_AM_VIB_EG_KSR_multiple(byte instnum, byte modulator, byte carrier) {
	al_out(choffs[instnum]+0x1f,modulator);
	al_out(choffs[instnum]+0x1f+3,carrier);
}

void mp_KSL_TotalLevel(byte instnum, byte modulator, byte carrier) {
	al_out(choffs[instnum]+0x3f,modulator);
	al_out(choffs[instnum]+0x3f+3,carrier);
}

void mp_Attack_Decay(byte instnum, byte modulator, byte carrier) {
	al_out(choffs[instnum]+0x5f,modulator);
	al_out(choffs[instnum]+0x5f+3,carrier);
}

void mp_Sustain_Release(byte instnum, byte modulator, byte carrier) {
	al_out(choffs[instnum]+0x7f,modulator);
	al_out(choffs[instnum]+0x7f+3,carrier);
}

void mp_Block_FNum(byte instnum, byte modulator, byte carrier) {
	al_out(instnum+0x9f,modulator);
	al_out(instnum+0xaf,carrier);
}

void mp_AMVIBDepth_Rhythm(byte data) {
	al_out(0xbd,data);
}

void mp_WaveSelect(byte instnum, byte modulator, byte carrier) {
	al_out(choffs[instnum]+0xdf,modulator);
	al_out(choffs[instnum]+0xdf+3,carrier);
}

void mp_Feedback_Connect(byte instnum, byte data) {
	al_out(instnum+0xbf,data);
}

void mp_LoadInst(byte instnum, TInstrument &instdata) {
	mp_AM_VIB_EG_KSR_Multiple(instnum,instdata[0],instdata[1]);
	mp_KSL_TotalLevel(instnum,instdata[2],instdata[3]);
	mp_Attack_Decay(instnum,instdata[4],instdata[5]);
	mp_Sustain_Release(instnum,instdata[6],instdata[7]);
	mp_WaveSelect(instnum,instdata[8],instdata[9]);
	mp_Feedback_Connect(instnum,instdata[10]);
}

void mp_NoteOn(byte instnum, byte octave, byte note) {
	word note2;
	byte b1,b2;

	note2 = fnums[note];
	b2 = (byte)(note2 >> 8)+0x20+(octave << 2);
	b1 = (byte)(note2 & 0x00ff);
	mp_Block_FNum(instnum,b1,b2);
}

void mp_NoteOff(byte instnum, byte octave, byte note) {
	word note2;
	byte b1,b2;

	note2 = fnums[note];
	b2 = (byte)(note2 >> 8)+(octave << 2);
	b1 = (byte)(note2 & 0x00ff);
	mp_Block_FNum(instnum,b1,b2);
}

void mp_setinstrument(int tracknum, int chnum, TInstrument *inst,
					  char percussive) {
	byte c, m;
	if (percussive) {
		c = pcarriers[chnum];
		m = pmodifiers[chnum];
	} else {
		c = carriers[chnum];
		m = modifiers[chnum];
	  }

	tracks[track-1]->inst = *inst;
	tracks[track-1]->percussive = percussive;

	alOut(m + alChar,inst->mChar);
	alOut(m + alScale,inst->mScale);
	alOut(m + alAttack,inst->mAttack);
	alOut(m + alSus,inst->mSus);
	alOut(m + alWave,inst->mWave);

	// Most percussive instruments only use one cell
	if (c != 0xff) {
		alOut(c + alChar,inst->cChar);
		alOut(c + alScale,inst->cScale);
		alOut(c + alAttack,inst->cAttack);
		alOut(c + alSus,inst->cSus);
		alOut(c + alWave,inst->cWave);
	}

	alOut(chnum + alFeedCon,inst->nFeedCon);	// DEBUG
}

void mp_MusicOn(void) {
	mp_mactive = 1;
}

void mp_MusicOff(void) {
{
	word i; !!!

	alFXReg = 0;
	alOut(alEffects,0);
	for (i = 0;i < sqMaxTracks;i++)
		alOut(alFreqH + i + 1,0);
	sm_mactive = 0;
}

void mp_StartMusic(MusicGroup far *music) {
	mp_MusicOff();
//asm	pushf
//asm	cli
	sqHackPtr = sqHack = music->values;
	sqHackSeqLen = sqHackLen = music->length;
	sqHackTime = 0;
	alTimeCount = 0;
	mp_MusicOn();
//asm	popf
}