/************************************************************************
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 ************************************************************************/

// CRC I/O - (C)1997  Valentini Domenico
// CRCio - I/O with 16 bit CRC checking, multiple channels

#include "std.hpp"
#include "mm4.hpp"
#include "fastmem.hpp"
#include "io.h"
#include "fcntl.h"
#include "crcio.hpp"

#define	CRC_MAXCHANNELS		4
#define	CRC_BUFLEN			8192

#define	CRC_validchannel(x)	(x>=0 && x<CRC_MAXCHANNELS)

#define	CRCS_IDLE		0
#define	CRCS_READING	1
#define	CRCS_WRITING 	2

static int CRC_channels = 0;
static int CRC_handle[CRC_MAXCHANNELS] =
		{-1,-1,-1,-1};
static word CRC_tot[CRC_MAXCHANNELS] =
		{0,0,0,0};
static char CRC_status[CRC_MAXCHANNELS] =
		{CRCS_IDLE,CRCS_IDLE,CRCS_IDLE,CRCS_IDLE};

void near CRC_reset(int channel) {
	CRC_tot[channel]=0;
}

void near CRC_add(int channel, void far *p, word size) {
asm {
	MOV BX, channel
	SHL BX, 1
	MOV DX, WORD PTR CRC_tot[BX]
	LDS SI, p
	XOR AX, AX
	MOV CX, size
	CLD
} c1: asm {
	LODSB
	ADD DX, AX
	LOOP c1
	// !!! SS==DS !!!
	MOV AX, SS
	MOV DS, AX
	MOV WORD PTR CRC_tot[BX], DX
}}

char near CRC_check(int channel) {
	int handle = CRC_handle[channel];
	if (handle<0) return 0;
	long initpos, size;
	void *p = mm_reserve(CRC_BUFLEN);
	word toread;
	CRC_reset(channel);
	size=filelength(handle)-(initpos = tell(handle))-2l; // 2  il CRC finale
	do {
		if (size>(long)CRC_BUFLEN) toread=CRC_BUFLEN; else toread=size;
		_read(handle,p,toread);
		CRC_add(channel,p,toread);
		size-=toread;
	} while (size);
	delete p;
	_read(handle,&toread,sizeof(toread));
	lseek(handle,initpos,SEEK_SET);
	return toread==CRC_tot[channel];
}

#pragma warn -rvl
int CRC_newchannel() {
	for (int i=0; i<CRC_MAXCHANNELS; i++)
		if (CRC_status[i]==CRCS_IDLE) {
			CRC_channels++;
			return i;
		}
	error("CRC_newchannel","fail");
	//return -1;
}
#pragma warn +rvl

int CRC_openread(const char *filename) {
	if (CRC_channels<CRC_MAXCHANNELS) {
		int handle = _open(filename,O_RDONLY|O_BINARY);
		if (handle>=0) {
			int channel = CRC_newchannel();
			CRC_handle[channel] = handle;
			CRC_status[channel] = CRCS_READING;
			if (CRC_check(channel))
				return channel;
			CRC_close(channel);
		}
	}
	return -1;
}

int CRC_openwrite(const char *filename) {
	if (CRC_channels<CRC_MAXCHANNELS) {
		int handle = _creat(filename,0x20/* FA_ARCH */);
		if (handle>=0) {
			int channel = CRC_newchannel();
			CRC_handle[channel] = handle;
			CRC_reset(channel);
			CRC_status[channel] = CRCS_WRITING;
			return channel;
		}
	}
	return -1;
}

int CRC_eof(int handle) {
	return eof(handle);
}

void CRC_read(int channel, void far *ptr, word size) {
	//if (CRC_validchannel(channel) && CRC_status[channel]==CRCS_READING) {
	if (size) _read(CRC_handle[channel],ptr,size);
	//} else error("Not reading");
}

void CRC_write(int channel, void far *ptr, word size) {
	//if (CRC_validchannel(channel) && CRC_status[channel]==CRCS_WRITING) {
	if (size) {
		_write(CRC_handle[channel],ptr,size);
		CRC_add(channel,ptr,size);
	}
	//} else error("Not writing");
}

void CRC_lseek(int channel, long displ, int from) {
	//if (CRC_validchannel(channel) && CRC_status[channel]==CRCS_READING) {
	lseek(CRC_handle[channel],displ,from);
	//else error("CRC_lseek - error");
}

void CRC_close(int channel) {
	if (!CRC_validchannel(channel) ||
		CRC_status[channel]==CRCS_IDLE) return;
	if (CRC_status[channel]==CRCS_WRITING)
		_write(CRC_handle[channel],&(CRC_tot[channel]),sizeof(CRC_tot[channel]));
	_close(CRC_handle[channel]);
	CRC_status[channel] = CRCS_IDLE;
	CRC_channels--;
}