/*      SCCS Id: @(#)u_init.c   3.1     93/07/07        */
/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1985. */
/* NetHack may be freely redistributed.  See license for details. */

#include "hack.h"

struct trobj {
	unsigned short int trotyp;
	schar trspe;
	char trclass;
	Bitfield(trquan,6);
	Bitfield(trknown,1);
	Bitfield(trbless,2);
};

static void FDECL(ini_inv, (struct trobj *));
static void FDECL(knows_object,(int));
static void FDECL(knows_class,(CHAR_P));
static int FDECL(role_index,(CHAR_P));

#define UNDEF_TYP       0
#define UNDEF_SPE       '\177'
#define UNDEF_BLESS     2

char elf_drow = FALSE; /* are you a drow instead of an elf? */

static boolean random_role = FALSE;

/* all roles must all have distinct first letter */
const char *roles[] = { /* also used in options.c and winxxx.c */
			/* roles[2] and [11] are changed for females */
			/* in all cases, the corresponding male and female */
			/* roles must start with the same letter */
	"Archeologist", "Barbarian", "Caveman", "Doppelganger", "Elf",
	"Flame Mage", "Gnome", "Healer", "Ice Mage", "Knight", "Lycanthrope",
	"Monk", "Necromancer", "Priest", "Rogue", "Samurai",
#ifdef TOURIST
	"Tourist",
#endif
	"Undead Slayer", "Valkyrie", "Wizard", 0

};

static struct trobj Cave_man[] = {
#define C_ARROWS        2
	{ CLUB, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ BOW, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ ARROW, 0, WEAPON_CLASS, 50, 1, UNDEF_BLESS }, /* quan is variable */
	{ LEATHER_ARMOR, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Barbarian[] = {
#define B_MAJOR 0       /* two-handed sword or battle-axe  */
#define B_MINOR 1       /* matched with axe or short sword */
	{ TWO_HANDED_SWORD, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ AXE, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ RING_MAIL, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 2, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Knight[] = {
	{ LONG_SWORD, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ SPEAR, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ PLATE_MAIL, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ HELMET, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ LARGE_SHIELD, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ LEATHER_GLOVES, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Doppelganger[] = {
#define D_RING          2
	{ SILVER_DAGGER, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 2, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, RING_CLASS, 1, 1, 1 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Elf[] = {
#define E_ARROWS_ONE    2
#define E_ARROWS_TWO    3
#define E_ARMOR         4
	{ ELVEN_SHORT_SWORD, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ ELVEN_BOW, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ ELVEN_ARROW, 0, WEAPON_CLASS, 50, 1, UNDEF_BLESS },
	{ ELVEN_ARROW, 2, WEAPON_CLASS, 50, 1, UNDEF_BLESS },
	{ UNDEF_TYP, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ LEMBAS_WAFER, 0, FOOD_CLASS, 4, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 2, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Drow[] = {
	{ DARK_ELVEN_SHORT_SWORD, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ DARK_ELVEN_BOW, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ DARK_ELVEN_ARROW, 0, WEAPON_CLASS, 50, 1, UNDEF_BLESS },
	{ DARK_ELVEN_MITHRIL_COAT, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ POT_SICKNESS, 0, POTION_CLASS, 2, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, POTION_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 3, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Flame_Mage[] = {
	{ QUARTERSTAFF, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ STUDDED_LEATHER_ARMOR, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 2, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, POTION_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, RING_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ SPE_FIREBALL, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ WAN_FIRE, UNDEF_SPE, WAND_CLASS, 1, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Ice_Mage[] = {
	{ QUARTERSTAFF, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ STUDDED_LEATHER_ARMOR, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 2, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, POTION_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, RING_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ SPE_CONE_OF_COLD, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ WAN_COLD, UNDEF_SPE, WAND_CLASS, 1, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Lycanthrope[] = {
	{ ORCISH_SHORT_SWORD, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ POT_SICKNESS, 0, POTION_CLASS, 2, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 2, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj UndeadSlayer[] = {
	{ SILVER_SPEAR, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ SILVER_DAGGER, 0, WEAPON_CLASS, 5, 1, UNDEF_BLESS },
	{ CHAIN_MAIL, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ HELMET, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ CLOVE_OF_GARLIC, 0, FOOD_CLASS, 5, 1, 1 },
	{ SPRIG_OF_WOLFSBANE, 0, FOOD_CLASS, 5, 1, 1 },
	{ HOLY_WAFER, 0, FOOD_CLASS, 4, 1, 0 },
	{ POT_WATER, 0, POTION_CLASS, 4, 1, 1 },        /* holy water */
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Valkyrie[] = {
	{ LONG_SWORD, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ DAGGER, 0, WEAPON_CLASS, 5, 1, UNDEF_BLESS },
	{ SMALL_SHIELD, 3, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Gnome[] = {
	{ AKLYS, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ DAGGER, 0, WEAPON_CLASS, 8, 1, UNDEF_BLESS },
	{ LEATHER_ARMOR, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ CRAM_RATION, 0, FOOD_CLASS, 3, 1, 0 },
	{ AGATE, 0, GEM_CLASS, 5, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Healer[] = {
	{ SCALPEL, 1, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ LEATHER_GLOVES, 1, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ STETHOSCOPE, 0, TOOL_CLASS, 1, 1, 0 },
	{ MEDICAL_KIT, UNDEF_SPE, TOOL_CLASS, 1, 1, 0 },
	{ POT_HEALING, 0, POTION_CLASS, 4, 1, UNDEF_BLESS },
	{ POT_EXTRA_HEALING, 0, POTION_CLASS, 4, 1, UNDEF_BLESS },
	{ WAN_SLEEP, UNDEF_SPE, WAND_CLASS, 1, 1, UNDEF_BLESS },
	/* [Tom] they might as well have a wand of healing, too */
	{ WAN_HEALING, UNDEF_SPE, WAND_CLASS, 1, 1, UNDEF_BLESS },
	{ SPE_HEALING, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ SPE_EXTRA_HEALING, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ APPLE, 0, FOOD_CLASS, 10, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Monk[] = {
#define M_BOOK          1
	{ LEATHER_GLOVES, 2, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 1, 1, UNDEF_BLESS },
	{ POT_HEALING, 0, POTION_CLASS, 3, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 3, 1, 0 },
	{ APPLE, 0, FOOD_CLASS, 5, 1, UNDEF_BLESS },
	{ ORANGE, 0, FOOD_CLASS, 5, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Archeologist[] = {
#define A_BOOK          4
	/* if adventure has a name...  idea from tan@uvm-gen */
	{ BULLWHIP, 2, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ LEATHER_JACKET, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FEDORA, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FOOD_RATION, 0, FOOD_CLASS, 3, 1, 0 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 2, 1, UNDEF_BLESS },
	{ PICK_AXE, UNDEF_SPE, TOOL_CLASS, 1, 1, UNDEF_BLESS },
	{ TINNING_KIT, 100, TOOL_CLASS, 1, 1, UNDEF_BLESS },
	{ SACK, 0, TOOL_CLASS, 1, 0, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Tinopener[] = {
	{ TIN_OPENER, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Magicmarker[] = {
	{ MAGIC_MARKER, UNDEF_SPE, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Lamp[] = {
	{ OIL_LAMP, 1, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Spellbook[] = {
	{ UNDEF_TYP, 1, SPBOOK_CLASS, 1, 1, 1 },
	{ 0, 0, 0, 0, 0, 0 }
};


#ifdef TOURIST
# ifdef WALKIES
static struct trobj Leash[] = {
	{ LEASH, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};
# endif

static struct trobj Towel[] = {
	{ TOWEL, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};
#endif

#ifdef EXPLORE_MODE
static struct trobj Wishing[] = {
	{ WAN_WISHING, 3, WAND_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};
#endif

static struct trobj Instrument[] = {
	{ WOODEN_FLUTE, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Blindfold[] = {
	{ BLINDFOLD, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

#ifdef TOURIST
static struct trobj Tourist[] = {
#define T_DARTS         0
	{ DART, 2, WEAPON_CLASS, 25, 1, UNDEF_BLESS },  /* quan is variable */
	{ UNDEF_TYP, UNDEF_SPE, FOOD_CLASS, 12, 1, 0 },
	{ POT_EXTRA_HEALING, 0, POTION_CLASS, 2, 1, UNDEF_BLESS },
	{ SCR_MAGIC_MAPPING, 0, SCROLL_CLASS, 6, 1, UNDEF_BLESS },
	{ HAWAIIAN_SHIRT, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ EXPENSIVE_CAMERA, 0, TOOL_CLASS, 1, 1, 0 },
	{ CREDIT_CARD, 0, TOOL_CLASS, 1, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};
#endif

static struct trobj Rogue[] = {
#define R_DAGGERS       1
#define R_DARTS         2
	{ SHORT_SWORD, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ DAGGER, 0, WEAPON_CLASS, 10, 1, UNDEF_BLESS },
	{ DART, 0, WEAPON_CLASS, 25, 1, UNDEF_BLESS },
	{ LEATHER_ARMOR, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ POT_SICKNESS, 0, POTION_CLASS, 2, 1, 0 },
	{ SCR_GOLD_DETECTION, 0, SCROLL_CLASS, 4, 1, 1 },
	{ SCR_TELEPORTATION, 0, SCROLL_CLASS, 4, 1, 1 },
	{ LOCK_PICK, 9, TOOL_CLASS, 1, 1, 0 },
	{ OILSKIN_SACK, 0, TOOL_CLASS, 1, 0, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Wizard[] = {
#define W_MULTSTART     2
#define W_MULTEND       5
#define W_BOOK1         6
#define W_BOOK2         7
#define W_BOOK3         8
#define W_BOOK4         9

	{ ATHAME, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },   /* for dealing with ghosts */
	{ CLOAK_OF_MAGIC_RESISTANCE, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, WAND_CLASS, 2, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, RING_CLASS, 2, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, POTION_CLASS, 4, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 4, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Necromancer[] = {
/* pretty much like Wizard, except with pick-axe instead of magic resist. */
	{ ATHAME, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },   /* for dealing with ghosts */
	{ UNDEF_TYP, UNDEF_SPE, WAND_CLASS, 2, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, RING_CLASS, 2, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, POTION_CLASS, 4, 1, UNDEF_BLESS },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 4, 1, UNDEF_BLESS },
	{ SPE_TURN_UNDEAD, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ SPE_SUMMON_UNDEAD, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ SPE_COMMAND_UNDEAD, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ PICK_AXE, UNDEF_SPE, TOOL_CLASS, 1, 1, UNDEF_BLESS },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Samurai[] = {
#define S_ARROWS        3
	{ KATANA, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ SHORT_SWORD, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS }, /* wakizashi */
	{ YUMI, 0, WEAPON_CLASS, 1, 1, UNDEF_BLESS },
	{ YA, 0, WEAPON_CLASS, 25, 1, UNDEF_BLESS }, /* variable quan */
	{ SPLINT_MAIL, 0, ARMOR_CLASS, 1, 1, UNDEF_BLESS },
	{ FORTUNE_COOKIE, 0, FOOD_CLASS, 3, 1, 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

static struct trobj Priest[] = {
#define P_BOOK          7
	{ MACE, 0, WEAPON_CLASS, 1, 1, 1 },
	{ CHAIN_MAIL, 0, ARMOR_CLASS, 1, 1, 1 },
	{ SMALL_SHIELD, 0, ARMOR_CLASS, 1, 1, 1 },
	{ POT_WATER, 0, POTION_CLASS, 4, 1, 1 },        /* holy water */
	{ CLOVE_OF_GARLIC, 0, FOOD_CLASS, 1, 1, 1 },
	{ SPRIG_OF_WOLFSBANE, 0, FOOD_CLASS, 1, 1, 1 },
	{ SPE_HEALING, 0, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SPBOOK_CLASS, 1, 1, 1 },
	{ UNDEF_TYP, UNDEF_SPE, SCROLL_CLASS, 1, 1, 1 },
	{ 0, 0, 0, 0, 0, 0 }
};

static void
knows_object(obj)
register int obj;
{
	makeknown(obj);
	objects[obj].oc_descr_idx = 0;          /* not a "discovery" */
}

/* Know ordinary (non-magical) objects of a certain class,
 * like all gems except the loadstone and luckstone.
 */
static void
knows_class(sym)
register char sym;
{
	register int ct;
	for (ct = 1; ct <= NROFOBJECTS; ct++)
		if (objects[ct].oc_class == sym && !objects[ct].oc_magic)
			knows_object(ct);
}

static int
role_index(pc)
char pc;
{
	register const char *cp;

	if ((cp = index(pl_classes, pc)) != 0)
		return(cp - pl_classes);
	return(-1);
}

void
u_init()
{
	register int i, temp, temp2;
	char pc;

	pc = pl_character[0];
	if(pc == '\0') {
	    /* should be unnecessary now */
	    exit_nhwindows(NULL);
	    terminate(0);
	}
	i = role_index(pc);
	if (random_role) {
	    pline("This game you will be %s.", an(roles[i]));
	    display_nhwindow(WIN_MESSAGE, TRUE);
	}

	(void) strncpy(pl_character, roles[i], PL_CSIZ-1);
	pl_character[PL_CSIZ-1] = 0;
	flags.beginner = 1;

	/* zero u, including pointer values --
	 * necessary when aborting from a failed restore */
	(void) memset((genericptr_t)&u, 0, sizeof(u));
	u.ustuck = (struct monst *)0;
	u.usick_cause = NULL;

#if 0   /* documentation of more zero values as desirable */
	u.uluck  = u.moreluck = 0;
# ifdef TOURIST
	uarmu = 0;
# endif
	uarm = uarmc = uarmh = uarms = uarmg = uarmf = 0;
	uwep = uball = uchain = uleft = uright = 0;
	u.ublessed = 0;                         /* not worthy yet */
	u.ugangr   = 0;                         /* gods not angry */
# ifdef ELBERETH
	u.uevent.uhand_of_elbereth = 0;
# endif
	u.uevent.uheard_tune = 0;
	u.uevent.uopened_dbridge = 0;
	u.uevent.udemigod = 0;          /* not a demi-god yet... */
	u.udg_cnt = 0;
# ifdef POLYSELF
	u.mh = u.mhmax = u.mtimedone = 0;
# endif
	u.uz.dnum = u.uz0.dnum = 0;
	u.utotype = 0;
#endif  /* 0 */
	u.uz.dlevel = u.uz0.dlevel = 1;
	u.utolev = u.uz;

	u.usym = S_HUMAN;
	u.umoved = FALSE;
	u.umortality = 0;
	u.ugrave_arise = -1;

	u.ulevel = 0;   /* set up some of the initial attributes */
	u.uhpmax = u.uhpbase = newhp();
	u.uhp = 25;
	adjabil(0,1);
	u.ulevel = 1;

	init_uhunger();
	u.uenbase = 1;
	u.uenmax = 1;
	u.uen = 25;
	for (i = 0; i <= MAXSPELL; i++) spl_book[i].sp_id = NO_SPELL;
	u.ublesscnt = 300;                      /* no prayers just yet */
#ifdef POLYSELF
	u.umonnum = -1;
	u.ulycn = -1;
	set_uasmon();
#endif

#ifdef BSD
	(void) time((long *)&u.ubirthday);
#else
	(void) time(&u.ubirthday);
#endif

	/*
	 *  For now, everyone but elves, cavemen and lycanthropes starts
	 *  out with a night vision range of 1 and their xray range disabled.
	 */
	u.nv_range   =  1;
	u.xray_range = -1;

/* [Tom] ask character alignment (for those non-specific types) */
	{ char align_answer = ' ';
	  if (pc == 'D' || pc == 'E' || pc == 'P' ||
	     pc == 'S') {
	     pline("Your alignment (L/N/C)?");
	     align_answer = readchar();
	     if (align_answer == 'l') u.ualign.type = A_LAWFUL;
	     if (align_answer == 'n') u.ualign.type = A_NEUTRAL;
	     if (align_answer == 'c') {
		u.ualign.type = A_CHAOTIC;
		if (pc == 'E') {
		   elf_drow = TRUE; /* global flag to mess with stuff */
		}
	     }
	  }
	  if (pc == 'B' || pc == 'R' || pc == 'V') {
	     pline("Your alignment (N/C)?");
	     align_answer = readchar();
	     if (align_answer == 'n') u.ualign.type = A_NEUTRAL;
	     if (align_answer == 'c') u.ualign.type = A_CHAOTIC;
	  }
	  if (pc == 'G' || pc == 'H' || pc == 'C' || pc == 'T' || pc == 'A') {
	     pline("Your alignment (L/N)?");
	     align_answer = readchar();
	     if (align_answer == 'l') u.ualign.type = A_LAWFUL;
	     if (align_answer == 'n') u.ualign.type = A_NEUTRAL;
	  }
	}

	switch(pc) {
	/* pc will always be in uppercase by this point */
	case 'A':
		u.umonster = PM_ARCHEOLOGIST;
		u.uen = u.uenmax = u.uenbase += 6;
		switch (rnd(5)) {   
		    case 1: Archeologist[A_BOOK].trotyp = SPE_DETECT_FOOD; break;
		    case 2: Archeologist[A_BOOK].trotyp = SPE_DETECT_MONSTERS; break;
		    case 3: Archeologist[A_BOOK].trotyp = SPE_LIGHT; break;
		    case 4: Archeologist[A_BOOK].trotyp = SPE_KNOCK; break;
		    case 5: Archeologist[A_BOOK].trotyp = SPE_WIZARD_LOCK; break;
		    default: break;
		}

		ini_inv(Archeologist);
		if(!rn2(4)) ini_inv(Blindfold);
		else if(!rn2(4)) ini_inv(Towel);
#ifdef WALKIES
		if(!rn2(4)) ini_inv(Leash);
#endif
		if(!rn2(4)) ini_inv(Tinopener);
		if(!rn2(4)) ini_inv(Lamp);
		if(!rn2(8)) ini_inv(Magicmarker);
		knows_class(GEM_CLASS);
		knows_object(SACK);
		/* We can't set trknown for it, then it'd be "uncursed"
		 * sack...
		 */
		break;
	case 'B':
		u.umonster = PM_BARBARIAN;
		if (rn2(100) >= 50) {   /* see Elf comment */
		    Barbarian[B_MAJOR].trotyp = BATTLE_AXE;
		    Barbarian[B_MINOR].trotyp = SHORT_SWORD;
		}
		ini_inv(Barbarian);
		if(!rn2(6)) ini_inv(Lamp);
		knows_class(WEAPON_CLASS);
		knows_class(ARMOR_CLASS);
		break;
	case 'C':
		u.nv_range = 2;
		u.umonster = PM_CAVEMAN;
		Cave_man[C_ARROWS].trquan = rn1(10, 50);
		ini_inv(Cave_man);
		break;
	case 'D':
		u.umonster = PM_DOPPELGANGER;
		/* [Tom] D's get such a high amount for their polymorphing...
		    (they suck at spells) */
		u.uen = u.uenmax = u.uenbase += 10;
		switch (rnd(16)) {
		    case 1:  Doppelganger[D_RING].trotyp = RIN_REGENERATION; break;
		    case 2:  Doppelganger[D_RING].trotyp = RIN_SEARCHING; break;
		    case 3:  Doppelganger[D_RING].trotyp = RIN_CONFLICT; break;
		    case 4:  Doppelganger[D_RING].trotyp = RIN_POISON_RESISTANCE; break;
		    case 5:  Doppelganger[D_RING].trotyp = RIN_FIRE_RESISTANCE; break;
		    case 6:  Doppelganger[D_RING].trotyp = RIN_COLD_RESISTANCE; break;
		    case 7:  Doppelganger[D_RING].trotyp = RIN_SHOCK_RESISTANCE; break;
		    case 8:  Doppelganger[D_RING].trotyp = RIN_TELEPORTATION; break;
		    case 9:  Doppelganger[D_RING].trotyp = RIN_TELEPORT_CONTROL; break;
		    case 10: Doppelganger[D_RING].trotyp = RIN_INVISIBILITY; break;
		    case 11: Doppelganger[D_RING].trotyp = RIN_SEE_INVISIBLE; break;
		    case 12: Doppelganger[D_RING].trotyp = RIN_PROTECTION;
			     Doppelganger[D_RING].trspe = rnd(5);
			     break;
		    case 13: Doppelganger[D_RING].trotyp = RIN_GAIN_STRENGTH;
			     Doppelganger[D_RING].trspe = rnd(5);
			     break;
		    case 14: Doppelganger[D_RING].trotyp = RIN_GAIN_CONSTITUTION;
			     Doppelganger[D_RING].trspe = rnd(5);
			     break;
		    case 15: Doppelganger[D_RING].trotyp = RIN_INCREASE_DAMAGE;
			     Doppelganger[D_RING].trspe = rnd(5);
			     break;
		    case 16: Doppelganger[D_RING].trotyp = RIN_ADORNMENT;
			     Doppelganger[D_RING].trspe = rnd(5);
			     break;
		    default: break;
		}
		ini_inv(Doppelganger);
		break;
	case 'E':
		u.nv_range = 2;
		u.uen = u.uenmax = u.uenbase += 6;
		if (!elf_drow) {
		u.umonster = PM_ELF;
		Elf[E_ARROWS_ONE].trquan = rn1(10, 50);
		Elf[E_ARROWS_TWO].trquan = rn1(10, 30);
		Elf[E_ARMOR].trotyp = ((rn2(100) >= 50)
				 ? ELVEN_MITHRIL_COAT : ELVEN_CLOAK);
			/* rn2(100) > 50 necessary because some random number
			 * generators are bad enough to seriously skew the
			 * results if we use rn2(2)...  --KAA
			 */
		ini_inv(Elf);
		/*
		 * Elves are people of music and song, or they are warriors.
		 * Warriors get mithril coats; non-warriors will get an
		 * instrument.  We use a kludge to get only non-magic
		 * instruments.
		 */
		if (Elf[E_ARMOR].trotyp == ELVEN_CLOAK) {
		    static int trotyp[] = {
			WOODEN_FLUTE, TOOLED_HORN, WOODEN_HARP,
			BELL, BUGLE, LEATHER_DRUM
		    };
		    Instrument[0].trotyp = trotyp[rn2(SIZE(trotyp))];
		    ini_inv(Instrument);
		    ini_inv(Spellbook);
		    if(!rn2(5)) ini_inv(Magicmarker);
		}
		if(!rn2(5)) ini_inv(Blindfold);
		else if(!rn2(6)) ini_inv(Lamp);
		knows_object(ELVEN_SHORT_SWORD);
		knows_object(ELVEN_ARROW);
		knows_object(ELVEN_BOW);
		knows_object(ELVEN_SPEAR);
		knows_object(ELVEN_DAGGER);
		knows_object(ELVEN_BROADSWORD);
		knows_object(ELVEN_MITHRIL_COAT);
		knows_object(ELVEN_LEATHER_HELM);
		knows_object(ELVEN_SHIELD);
		knows_object(ELVEN_BOOTS);
		knows_object(ELVEN_CLOAK);
		} else { /* is a dark elf */
		   u.umonster = PM_DROW;
		   knows_object(DARK_ELVEN_SHORT_SWORD);
		   knows_object(DARK_ELVEN_ARROW);
		   knows_object(DARK_ELVEN_BOW);
		   knows_object(DARK_ELVEN_DAGGER);
		   knows_object(DARK_ELVEN_MITHRIL_COAT);
		   ini_inv(Drow);
		}
		break;
	case 'F':
		u.umonster = PM_FLAME_MAGE;
		u.uen = u.uenmax = u.uenbase += 6;
		ini_inv(Flame_Mage);
		if(!rn2(5)) ini_inv(Lamp);
		else if(!rn2(5)) ini_inv(Blindfold);
		else if(!rn2(5)) ini_inv(Magicmarker);
		break;
	case 'G':
		u.nv_range = 2;
		u.umonster = PM_GNOME;
		ini_inv(Gnome);
		knows_class(GEM_CLASS);
		break;
	case 'H':
		u.umonster = PM_HEALER;
		u.uen = u.uenmax = u.uenbase += 10;
		u.ugold = u.ugold0 = rn1(1000, 1001);
		ini_inv(Healer);
		knows_class(POTION_CLASS);
		knows_object(POT_SICKNESS);
		knows_object(POT_BLINDNESS);
		knows_object(POT_HALLUCINATION);
		knows_object(POT_RESTORE_ABILITY);
		if(!rn2(5)) ini_inv(Lamp);
		if(!rn2(5)) ini_inv(Magicmarker);
		if(!rn2(5)) ini_inv(Blindfold);
		break;
	case 'I':
		u.umonster = PM_ICE_MAGE;
		u.uen = u.uenmax = u.uenbase += 6;
		ini_inv(Ice_Mage);
		if(!rn2(5)) ini_inv(Lamp);
		else if(!rn2(5)) ini_inv(Blindfold);
		else if(!rn2(5)) ini_inv(Magicmarker);
		break;
	case 'K':
		u.umonster = PM_KNIGHT;
		u.uen = u.uenmax = u.uenbase += 3;
		ini_inv(Knight);
		knows_class(WEAPON_CLASS);
		knows_class(ARMOR_CLASS);
		/* give knights chess-like mobility
		 * -- idea from wooledge@skybridge.scl.cwru.edu */
		Jumping |= FROMOUTSIDE;
		break;
	case 'L':
		u.nv_range = 2;
		u.ulycn = PM_WEREWOLF;
		u.uen = u.uenmax = u.uenbase += 6;
		u.umonster = PM_HUMAN_WEREWOLF;
		ini_inv(Lycanthrope);
		if(!rn2(5)) ini_inv(Blindfold);
		break;
	case 'M':
		u.umonster = PM_MONK;
		u.uen = u.uenmax = u.uenbase += 6;
		switch (rnd(3)) {
		    case 1: Monk[M_BOOK].trotyp = SPE_HEALING; break;
		    case 2: Monk[M_BOOK].trotyp = SPE_FORCE_BOLT; break;
		    case 3: Monk[M_BOOK].trotyp = SPE_SLEEP; break;
		    default: break;
		}
		ini_inv(Monk);
		if(!rn2(10)) ini_inv(Magicmarker);
		else if(!rn2(10)) ini_inv(Lamp);
		knows_class(ARMOR_CLASS);
		break;
	case 'N':
		u.umonster = PM_NECROMANCER;
		u.uen = u.uenmax= u.uenbase += 10;
		ini_inv(Necromancer);
		knows_class(SPBOOK_CLASS);
		if(!rn2(5)) ini_inv(Magicmarker);
		if(!rn2(5)) ini_inv(Blindfold);
		break;

	case 'P':
		u.umonster = PM_PRIEST;
		u.uen = u.uenmax = u.uenbase += 10;
		switch (rnd(9)) {   
		    case 1: Priest[P_BOOK].trotyp = SPE_FORCE_BOLT; break;
		    case 2: Priest[P_BOOK].trotyp = SPE_SLEEP; break;
		    case 3: Priest[P_BOOK].trotyp = SPE_RESIST_POISON; break;
		    case 4: Priest[P_BOOK].trotyp = SPE_RESIST_SLEEP; break;
		    case 5: Priest[P_BOOK].trotyp = SPE_DETECT_FOOD; break;
		    case 6: Priest[P_BOOK].trotyp = SPE_DETECT_MONSTERS; break;
		    case 7: Priest[P_BOOK].trotyp = SPE_LIGHT; break;
		    case 8: Priest[P_BOOK].trotyp = SPE_KNOCK; break;
		    case 9: Priest[P_BOOK].trotyp = SPE_WIZARD_LOCK; break;
		    default: break;
		}
		ini_inv(Priest);
		if(!rn2(10)) ini_inv(Magicmarker);
		else if(!rn2(10)) ini_inv(Lamp);
		knows_object(POT_WATER);
		break;
	case 'R':
		u.umonster = PM_ROGUE;
		u.uen = u.uenmax = u.uenbase += 3;
		Rogue[R_DAGGERS].trquan = rn1(10, 6);
		Rogue[R_DARTS].trquan = rn1(10, 25);
		u.ugold = u.ugold0 = rn1(500,1500);
		ini_inv(Rogue);
		if(!rn2(5)) ini_inv(Blindfold);
		knows_object(OILSKIN_SACK);
		break;
	case 'S':
		u.umonster = PM_SAMURAI;
		Samurai[S_ARROWS].trquan = rn1(20, 26);
		ini_inv(Samurai);
		if(!rn2(5)) ini_inv(Blindfold);
		knows_class(WEAPON_CLASS);
		knows_class(ARMOR_CLASS);
		break;
#ifdef TOURIST
	case 'T':
		u.umonster = PM_TOURIST;
		u.uen = u.uenmax = u.uenbase += 3;
		Tourist[T_DARTS].trquan = rn1(20, 21);
		u.ugold = u.ugold0 = rn1(500,1000);
		ini_inv(Tourist);
		if(!rn2(25)) ini_inv(Tinopener);
#ifdef WALKIES
		else if(!rn2(25)) ini_inv(Leash);
#endif
		else if(!rn2(25)) ini_inv(Towel);
		else if(!rn2(25)) ini_inv(Magicmarker);
		break;
#endif

	case 'U':
		u.umonster = PM_UNDEAD_SLAYER;
		u.uen = u.uenmax = u.uenbase += 3;
		ini_inv(UndeadSlayer);
		knows_class(WEAPON_CLASS);
		knows_class(ARMOR_CLASS);
		if(!rn2(6)) ini_inv(Lamp);
		break;
	case 'V':
		u.umonster = PM_VALKYRIE;
		flags.female = TRUE;
		ini_inv(Valkyrie);
		if(!rn2(6)) ini_inv(Lamp);
		knows_class(WEAPON_CLASS);
		knows_class(ARMOR_CLASS);
		break;
	case 'W':
		u.umonster = PM_WIZARD;
		u.uen = u.uenmax= u.uenbase += 10;
		switch (rnd(2)) {
		    case 1: Wizard[W_BOOK1].trotyp = SPE_FORCE_BOLT; break;
		    case 2: Wizard[W_BOOK1].trotyp = SPE_SLEEP; break;
		    default: break;
		}
		switch (rnd(2)) {
		    case 1: Wizard[W_BOOK2].trotyp = SPE_RESIST_POISON; break;
		    case 2: Wizard[W_BOOK2].trotyp = SPE_RESIST_SLEEP; break;
		    default: break;
		}
		switch (rnd(5)) {   
		    case 1: Wizard[W_BOOK3].trotyp = SPE_DETECT_FOOD; break;
		    case 2: Wizard[W_BOOK3].trotyp = SPE_DETECT_MONSTERS; break;
		    case 3: Wizard[W_BOOK3].trotyp = SPE_LIGHT; break;
		    case 4: Wizard[W_BOOK3].trotyp = SPE_KNOCK; break;
		    case 5: Wizard[W_BOOK3].trotyp = SPE_WIZARD_LOCK; break;
		    default: break;
		}
		switch (rnd(9)) {
		    case 1: Wizard[W_BOOK4].trotyp = SPE_MAGIC_MISSILE; break;
		    case 2: Wizard[W_BOOK4].trotyp = SPE_CONFUSE_MONSTER; break;
		    case 3: Wizard[W_BOOK4].trotyp = SPE_SLOW_MONSTER; break;
		    case 4: Wizard[W_BOOK4].trotyp = SPE_CURE_BLINDNESS; break;
		    case 5: Wizard[W_BOOK4].trotyp = SPE_ENDURE_HEAT; break;
		    case 6: Wizard[W_BOOK4].trotyp = SPE_ENDURE_COLD; break;
		    case 7: Wizard[W_BOOK4].trotyp = SPE_INSULATE; break;
		    case 8: Wizard[W_BOOK4].trotyp = SPE_CREATE_MONSTER; break;
		    case 9: Wizard[W_BOOK4].trotyp = SPE_HEALING; break;
		    default: break;
		}
		ini_inv(Wizard);
		knows_class(SPBOOK_CLASS);
		if(!rn2(5)) ini_inv(Magicmarker);
		if(!rn2(5)) ini_inv(Blindfold);
		break;

	default:        /* impossible */
		break;
	}

/* [Tom] ask character sex */
	if (pc != 'V') {
		char sex_answer;
		flags.female = TRUE;
		pline("Your sex (m/f) [m]?");
		sex_answer = readchar();
		if (sex_answer != 'f') {
			flags.female = FALSE;
		}
	}



#ifdef EXPLORE_MODE
	if (discover)
		ini_inv(Wishing);
#endif
	u.ugold0 += hidden_gold();      /* in case sack has gold in it */

	find_ac();                      /* get initial ac value */
	temp = rn1(10,70);
	init_attr(temp);                 /* init attribute values */
	max_rank_sz();                  /* set max str size for class ranks */
/*
 *      Do we really need this?
 */
	for(i = 0; i < A_MAX; i++)
	    if(!rn2(20)) {
		register int xd = rn2(7) - 2;   /* biased variation */
		(void) adjattrib(i, xd, TRUE);
		if (ABASE(i) < AMAX(i)) AMAX(i) = ABASE(i);
	    }
/* STEPHEN WHITE'S NEW CODE */
	/* give bigger percent strength above 18, instead of just a point */

	if (AMAX(A_STR) > 18) {
		temp = (18 + d((AMAX(A_STR) - 18), 10) +
				(2 * (AMAX(A_STR) - 18)));
		switch  (pl_character[0]) {
			case 'A':   temp2 = CLASS_A(A_STR); break;
			case 'B':   temp2 = CLASS_B(A_STR); break;
			case 'C':   temp2 = CLASS_C(A_STR); break;
			case 'E':   temp2 = CLASS_E(A_STR); break;
			case 'F':   temp2 = CLASS_F(A_STR); break;
			case 'G':   temp2 = CLASS_G(A_STR); break;
			case 'H':   temp2 = CLASS_H(A_STR); break;
			case 'I':   temp2 = CLASS_I(A_STR); break;
			case 'K':   temp2 = CLASS_K(A_STR); break;
			case 'L':   temp2 = CLASS_L(A_STR); break;
			case 'M':   temp2 = CLASS_M(A_STR); break;
			case 'N':   temp2 = CLASS_N(A_STR); break;
			case 'P':   temp2 = CLASS_P(A_STR); break;
			case 'R':   temp2 = CLASS_R(A_STR); break;
			case 'S':   temp2 = CLASS_S(A_STR); break;
		#ifdef TOURIST
			case 'T':   temp2 = CLASS_T(A_STR); break;
		#endif
			case 'U':   temp2 = CLASS_U(A_STR); break;
			case 'V':   temp2 = CLASS_V(A_STR); break;
			case 'W':   temp2 = CLASS_W(A_STR); break;
			default:    /* unknown type */
				    temp2 = CLASS_A(A_STR); break;
			}
		
		if (temp > temp2) temp = temp2;
		ABASE(A_STR) = temp;
		AMAX(A_STR) = temp;
		}

	/* make sure you can carry all you have - especially for Tourists */
	while(inv_weight() > 0 && ACURR(A_STR) < 118)
		(void) adjattrib(A_STR, 1, TRUE);

	u.ualignbase[0] = u.ualignbase[1] = u.ualign.type;
	u.uhp = 25;
	u.uen = 25;
}

static void
ini_inv(trop)
register struct trobj *trop;
{
	struct obj *obj;
	while(trop->trclass) {
		boolean undefined = (trop->trotyp == UNDEF_TYP);

		if (!undefined)
			obj = mksobj((int)trop->trotyp, TRUE, FALSE);
		else obj = mkobj(trop->trclass,FALSE);

		/* For random objects, do not create certain overly powerful
		 * items: wand of wishing, ring of levitation, or the
		 * polymorph/polymorph control combination.  Specific objects,
		 * i.e. the discovery wishing, are still OK.
		 * Also, don't get a couple of really useless items.  (Note:
		 * punishment isn't "useless".  Some players who start out with
		 * one will immediately read it and use the iron ball as a
		 * weapon.)
		 */
		if (undefined) {
#ifdef POLYSELF
			static NEARDATA unsigned nocreate = STRANGE_OBJECT;
			static NEARDATA unsigned nocreate2 = STRANGE_OBJECT;
#endif
			static NEARDATA unsigned nocreate3 = STRANGE_OBJECT;
			static NEARDATA unsigned nocreate4 = STRANGE_OBJECT;
			static NEARDATA unsigned nocreate5 = STRANGE_OBJECT;
			static NEARDATA unsigned nocreate6 = STRANGE_OBJECT;
			static NEARDATA unsigned nocreate7 = STRANGE_OBJECT;

			while(obj->otyp == WAN_WISHING
#ifdef POLYSELF
				|| obj->otyp == nocreate
				|| obj->otyp == nocreate2
#endif
				|| obj->otyp == nocreate3
				|| obj->otyp == nocreate4
				|| obj->otyp == nocreate5
				|| obj->otyp == nocreate6
				|| obj->otyp == nocreate7
#ifdef ELBERETH
				|| obj->otyp == RIN_LEVITATION
#endif
				|| ((pl_character[0] == 'F' ||
				     pl_character[0] == 'I')
						&&
				    (obj->otyp == RIN_FIRE_RESISTANCE || 
				     obj->otyp == RIN_COLD_RESISTANCE)) 

				|| (pl_character[0] == 'N' && obj->otyp == RIN_DRAIN_RESISTANCE)
				/* 'useless' or over powerful items */
				|| obj->otyp == POT_HALLUCINATION
				|| obj->otyp == POT_BLINDNESS
				|| obj->otyp == POT_WATER
				|| obj->otyp == POT_FRUIT_JUICE
				|| obj->otyp == SCR_BLANK_PAPER
				|| obj->otyp == SCR_GENOCIDE
				|| obj->otyp == SCR_AMNESIA
				|| obj->otyp == SCR_FIRE
				|| obj->otyp == SCR_PUNISHMENT
				|| obj->otyp == RIN_AGGRAVATE_MONSTER
				|| obj->otyp == RIN_HUNGER
				|| obj->otyp == RIN_SLEEPING
				|| obj->otyp == WAN_NOTHING
				/* powerful spells are either useless to
				   low level players or unbalancing */
				|| (obj->oclass == SPBOOK_CLASS &&
				    objects[obj->otyp].oc_level > 2)
							) {
				dealloc_obj(obj);
				obj = mkobj(trop->trclass, FALSE);
			}

			/* Don't start with +0 or negative rings */
			if(objects[obj->otyp].oc_charged && obj->spe <= 0)
				obj->spe = rnd(3);

			/* Heavily relies on the fact that 1) we create wands
			 * before rings, 2) that we create rings before
			 * spellbooks, and that 3) not more than 1 object of a
			 * particular symbol is to be prohibited.  (For more
			 * objects, we need more nocreate variables...)
			 */
#ifdef POLYSELF
			switch (obj->otyp) {
			    case WAN_POLYMORPH:
			    case RIN_POLYMORPH:
				nocreate = RIN_POLYMORPH_CONTROL;
				break;
			    case RIN_POLYMORPH_CONTROL:
				nocreate = RIN_POLYMORPH;
				nocreate2 = SPE_POLYMORPH;
			}
#endif /* POLYSELF */
			/* Don't have 2 of the same ring */
			if (obj->oclass == RING_CLASS)
				nocreate3 = obj->otyp;
			if (obj->oclass == SPBOOK_CLASS) {
				if (nocreate4 == STRANGE_OBJECT)
				    nocreate4 = obj->otyp;
				else if (nocreate5 == STRANGE_OBJECT)
				    nocreate5 = obj->otyp;
				else if (nocreate6 == STRANGE_OBJECT)
				    nocreate6 = obj->otyp;
				else nocreate7 = obj->otyp;
				}
		}

		obj->known = obj->dknown = obj->bknown = obj->rknown = trop->trknown;
		
		if(objects[obj->otyp].oc_uses_known) obj->known = trop->trknown;
		
		/* not obj->dknown = 1; - let him look at it at least once */
		obj->cursed = 0;

		if (obj->opoisoned && u.ualign.type != A_CHAOTIC) obj->opoisoned = 0;

		if(obj->oclass == WEAPON_CLASS || obj->oclass == TOOL_CLASS) {
			obj->quan = (long) trop->trquan;
			trop->trquan = 1;
		}
		
		if(obj->oclass == FOOD_CLASS && undefined) {
			obj->known = 1;
			/* needed for tins and eggs; harmless otherwise */
			obj->bknown = 1;
		}
		
		/*
		 * The below lines not needed because they don't correspond
		 * to any actual inventory; nobody gets random tools.
		else if(obj->oclass == TOOL_CLASS && undefined) {
			obj->bknown = (obj->otyp != BAG_OF_TRICKS
				&& obj->otyp != SACK
				&& obj->otyp != CHEST
				&& obj->otyp != LARGE_BOX
				&& obj->otyp != ICE_BOX);
		}
		*/
		if(trop->trspe != UNDEF_SPE)
			obj->spe = trop->trspe;

		if(trop->trbless != UNDEF_BLESS)
			obj->blessed = trop->trbless;

		/* defined after setting otyp+quan + blessedness */
		obj->owt = weight(obj);
				
		obj = addinv(obj);

		/* Make the type known if necessary */
		if (OBJ_DESCR(objects[obj->otyp]) && obj->known)
			makeknown(obj->otyp);
		
		if(obj->oclass == ARMOR_CLASS){
			if (is_shield(obj) && !uarms)
				setworn(obj, W_ARMS);
			else if (is_helmet(obj) && !uarmh)
				setworn(obj, W_ARMH);
			else if (is_gloves(obj) && !uarmg)
				setworn(obj, W_ARMG);
#ifdef TOURIST
			else if (obj->otyp == HAWAIIAN_SHIRT && !uarmu)
				setworn(obj, W_ARMU);
#endif
			else if (is_cloak(obj) && !uarmc)
				setworn(obj, W_ARMC);
			else if (is_boots(obj) && !uarmf)
				setworn(obj, W_ARMF);
			else if (!uarm)
				setworn(obj, W_ARM);
		}
		/* below changed by GAN 01/09/87 to allow wielding of
		 * pick-axe or can-opener if there is no weapon
		 */

		if(obj->oclass == WEAPON_CLASS || obj->otyp == PICK_AXE ||
		   obj->otyp == TIN_OPENER)
			if(!uwep) setuwep(obj);
		
#if !defined(PYRAMID_BUG) && !defined(MAC)
		if(--trop->trquan) continue;    /* make a similar object */
#else
		if(trop->trquan) {              /* check if zero first */
			--trop->trquan;
			if(trop->trquan)
				continue;       /* make a similar object */
		}
#endif
		
		trop++;
		
	}
	
}

void
plnamesuffix()
{
	register char *p;
	if ((p = rindex(plname, '-')) != 0) {
		*p = '\0';
		pl_character[0] = p[1];
		pl_character[1] = '\0';
		random_role = FALSE;
		if(!plname[0]) {
			askname();
			plnamesuffix();
		}
	}
	if (pl_character[0] == '@') {   /* explicit request for random class */
		int i = rn2((int)strlen(pl_classes));
		pl_character[0] = pl_classes[i];
		pl_character[1] = '\0';
		random_role = TRUE;
	}
}

/*u_init.c*/



